package com.futronictech.fs28;

import java.io.IOException;
import java.io.OutputStream;
import java.util.Timer;
import java.util.TimerTask;

import javax.microedition.io.Connector;
import javax.microedition.io.file.FileConnection;

import com.futronictech.fs28.FS28Demo;

import net.rim.device.api.bluetooth.BluetoothSerialPort;
import net.rim.device.api.bluetooth.BluetoothSerialPortInfo;
import net.rim.device.api.bluetooth.BluetoothSerialPortListener;
import net.rim.device.api.system.Bitmap;
import net.rim.device.api.ui.MenuItem;
import net.rim.device.api.ui.component.BitmapField;
import net.rim.device.api.ui.component.Dialog;
import net.rim.device.api.ui.component.LabelField;
import net.rim.device.api.ui.component.Status;
import net.rim.device.api.ui.container.MainScreen;
import net.rim.device.api.util.Arrays;
import net.rim.device.api.util.DataBuffer;

public class SPPScreen  extends MainScreen implements BluetoothSerialPortListener{

	public static final int DATA_TYPE_RAWIMAGE = 0; 
	public static final int DATA_TYPE_WSQIMAGE = 1; 
	public static final int DATA_TYPE_SAMPLE = 2;

    public static final int TIMEOUT_RAW = 20000;  // 20sec timeout
    public static final int TIMEOUT_OTHERS = 3000;  // 3sec timeout
	
    private byte[] _receiveBuffer = new byte[2048];
    private BluetoothSerialPort _port;
    private static boolean _dataSent = true;
    private String _deviceName;
    private DataBuffer _db;
    private LabelField _lblMessage;
    private BitmapField _bmpFP;
    // received data is 13 bytes command + n bytes data (wsq or raw)
    // n = command[6-9]
    // command[1] == 0x36 - start to convert wsq
    // command[1] == 0x0F - download wsq data
    // command[1] == 0x44 - download raw data
    // command[1] == 0x4D - download sample data
    private byte[] mCommand = new byte[13];
    private int mCommandLength = 0;
    private long mTimeDownloadImage = 0;
    private int mDataType = DATA_TYPE_RAWIMAGE;
    private int mbytesRead = 0; 
    private int mbytesTotal = 153602;
    private int mbytesTotalRead = 0;
    private byte[] mImageBuffer = new byte[153602];
    private MyBitmapFile mFileBmp = null;
    private int mWSQDataSize = 0;
    private boolean mDataMode = false;
    private boolean mDataReceived = false;
    private String mErrorMessage = null;   
    private byte[] mHostSample = new byte[669];  
    private Timer mTimer = null;
    private TimerTask mTimerTask = null;

    /**
     * Constructs a serial port connection screen.
     * @param info The information describing the bluetooth serial port to connect to. If 'null' then a new serial port is opened on this device.
     */
    public SPPScreen(BluetoothSerialPortInfo info)
    {
        Arrays.fill(_receiveBuffer, (byte)0);
        // Initialize the buffers
        _db = new DataBuffer();
        _lblMessage = new LabelField("", LabelField.USE_ALL_WIDTH);
        add(_lblMessage);
        Bitmap bitmapImage = Bitmap.getBitmapResource("logo.png");
        _bmpFP = new BitmapField(bitmapImage);             
        add(_bmpFP);
        try
        {
            if(info == null)
            {
                // Open a port to listen for incoming connections
                //For stable connection, use baudrate 115200.
            	_port = new BluetoothSerialPort("Hi there", BluetoothSerialPort.BAUD_115200 , BluetoothSerialPort.DATA_FORMAT_PARITY_NONE | BluetoothSerialPort.DATA_FORMAT_STOP_BITS_1 | BluetoothSerialPort.DATA_FORMAT_DATA_BITS_8, BluetoothSerialPort.FLOW_CONTROL_NONE, 2048, 2048, this);
            	_lblMessage.setText("Waiting for connections...");
                _deviceName = "unknown"; 
            }
            else
            {
                // Connect to the selected device
            	//For stable connection, use baudrate 115200.
            	_port = new BluetoothSerialPort(info, BluetoothSerialPort.BAUD_115200, BluetoothSerialPort.DATA_FORMAT_PARITY_NONE | BluetoothSerialPort.DATA_FORMAT_STOP_BITS_1 | BluetoothSerialPort.DATA_FORMAT_DATA_BITS_8, BluetoothSerialPort.FLOW_CONTROL_NONE, 2048, 2048, this);
                _deviceName = info.getDeviceName();
            }
        }
        catch(IOException ex)
        {
        	FS28Demo.errorDialog(ex.toString());
        }

        // Add menu items to the screen
        addMenuItem(_saveFile);
        addMenuItem(_closeSP);
        
        mTimer = new Timer();
    }

    /**
     * Alerts the user if a device was connected or if it failed to connect
     * @param success True if the connection was successful, false otherwise
     * @see net.rim.device.api.bluetooth.BluetoothSerialPortListener#deviceConnected(boolean) 
     */
    public void deviceConnected(boolean success)
    {
        if (success)
        {
            Status.show("Bluetooth SPP connected");
            _lblMessage.setText("Connected");
        }
        else
        {
            Status.show("Bluetooth SPP failed to connect");
            _lblMessage.setText("Failed to connect");
        }
    }

    /**
     * @see net.rim.device.api.bluetooth.BluetoothSerialPortListener#deviceDisconnected() 
     */
    public void deviceDisconnected()
    {
        Status.show("Disconnected from " + _deviceName);
    }
    /**
     * @see net.rim.device.api.bluetooth.BluetoothSerialPortListener#dtrStateChange(boolean) 
     */
    public void dtrStateChange(boolean high)
    {
        //Status.show("DTR: " + high);
    }
    /**
     * @see net.rim.device.api.bluetooth.BluetoothSerialPortListener#dataReceived(int) 
     */
    public void dataReceived(int length)
    {
        if( !mDataMode )
        {
            // Read the 13 bytes command data
        	if( length > 13 )
        		return;
            try
            {
                if((mbytesRead = _port.read(_receiveBuffer, 0, length == -1 ? 13 : length)) <= 0)
                	return;
            } catch(IOException ioex)
            {
            	FS28Demo.errorDialog(ioex.toString());
            	return;
            }
            // 
            System.arraycopy(_receiveBuffer, 0, mCommand, mCommandLength, mbytesRead);
        	mCommandLength += mbytesRead;
        	if(mCommandLength == 13)
        	{
				mbytesTotalRead = 0;
				mCommandLength = 0;	
    			int[] Size = new int[1];                    		
    			if( !validateCommandData(mCommand, Size) )
    			{
    				responseToFS28((byte)0x00, (byte)0x41);
    				FS28Demo.errorDialog(mErrorMessage);
    				return;
    			}    
    			switch (mCommand[1])
    			{
    			case 0x36: // 0x36 command means FS28 start to convert WSQ
    				responseToFS28((byte)0x00, (byte)0x40);	//reply OK
    				_lblMessage.setText("Got convert WSQ cmd");
    				return;
    			case 0x0F: //0x0F command means FS28 start to download WSQ
    				responseToFS28((byte)0x00, (byte)0x40);	//reply OK
    				mbytesTotal = Size[0];
    				mWSQDataSize = mbytesTotal - 2;
    				_lblMessage.setText("Downloading WSQ image");
    				mDataType = DATA_TYPE_WSQIMAGE;
    				mTimeDownloadImage = System.currentTimeMillis();
    				startTimer(TIMEOUT_OTHERS);
    				break;
    			case 0x44: //0x44 command means FS28 start to download RAW image, the RAW image size is always 153602 byte
    				responseToFS28((byte)0x00, (byte)0x40);	//reply OK
    				mbytesTotal = 153602;
    				mDataType = DATA_TYPE_RAWIMAGE;
    				_lblMessage.setText("Downloading RAW image");
    				mTimeDownloadImage = System.currentTimeMillis();
    				startTimer(TIMEOUT_RAW);
    				break;
    			case 0x4D: //0x4D command means FS28 start to download sample, the sample size is always 666 bytes
    				responseToFS28((byte)0x00, (byte)0x40);	//reply OK
    				mDataType = DATA_TYPE_SAMPLE;
    				mbytesTotal = 666;
    				_lblMessage.setText("Downloading Sample data");
    				mTimeDownloadImage = System.currentTimeMillis();
    				startTimer(TIMEOUT_OTHERS);
    				break;
    			default:
    				_lblMessage.setText("Unknown command");
    				responseToFS28((byte)0x00, (byte)0x41);
    				return;
    			}
	            if (mbytesTotal > 153602)
	            {
	            	_lblMessage.setText("Data size is too large! - " + mbytesTotal);
    				responseToFS28((byte)0x00, (byte)0x41);
	                return;
	            }                    			
	            mDataMode = true;
	            mDataReceived = false;
                return;
        	}
        	else
        		return;
        }
        //DataMode which is receiving WSQ/RAW image data from FS28, no command anymore.
        if( length > _receiveBuffer.length )
        	length = _receiveBuffer.length;
        try
        {
            if((mbytesRead = _port.read(_receiveBuffer, 0, length == -1 ? _receiveBuffer.length : length)) <= 0)
            {
            	return;
            }
        } catch(IOException ioex)
        {
        	FS28Demo.errorDialog(ioex.toString());
        	return;
        }
  
        System.arraycopy(_receiveBuffer, 0, mImageBuffer, mbytesTotalRead, mbytesRead);
        mbytesTotalRead += mbytesRead;
        _lblMessage.setText("Downloading ...... " + (mbytesTotalRead * 100 / mbytesTotal) + "%" );
        if (mbytesTotalRead >= mbytesTotal)// finish receiving all the WSQ/RAW image data
        {
        	stopTimer();
        	responseToFS28((byte)0x00, (byte)0x40);	//reply OK, finished receiving data
            enterCommandMode();	//return to Command mode 
            mDataReceived = true;
        	String strMsg = null;
        	if( mDataType == DATA_TYPE_RAWIMAGE )
        		strMsg = "RAW image downloaded! Time: ";
        	else if( mDataType == DATA_TYPE_WSQIMAGE )
        		strMsg = "WSQ image downloaded! Time: ";
        	else if ( mDataType == DATA_TYPE_SAMPLE )
        		strMsg = "Sample downloaded! Time: ";        	
        	_lblMessage.setText(strMsg + (System.currentTimeMillis() - mTimeDownloadImage) + "ms");
            showBitmap();
        }
  }

    /**
     * @see net.rim.device.api.bluetooth.BluetoothSerialPortListener#dataSent() 
     */
    public void dataSent()
    {
        // Set the _dataSent flag to true to allow more data to be written
        _dataSent = true;

        // Call sendData in case there is data waiting to be sent
        sendData();
    }

    /**
     * Writes a byte to the 'send' buffer
     * @param theData The byte to write to the buffer
     *
    private void writeData(byte theData)
    {
        synchronized(_db)
        {
            _db.write(theData);

            // Call sendData to send the data
            sendData();
        }
    }
	*/
    /**
     * Writes data from a source byte array to the 'send' buffer 
     * @param theData The source data to write to the buffer
     * @param offset The off set in the source data to write to the buffer
     * @param length The length of the array of bytes to write to the buffer
     */
    private void writeData(byte[] theData, int offset, int length)
    {
        synchronized(_db)
        {
            _db.write(theData, offset, length);

            // Call sendData to send the data
            sendData();
        }
    }

    /**
     * Sends the data currently stored in the DataBuffer to the other device
     */
    private void sendData()
    {
        // Ensure we have data to send
        if (_db.getArrayLength() > 0)
        {
            // Ensure the last write call has resulted in the sending of the data
            // prior to calling write again.  Calling write in sequence without waiting
            // for the data to be sent can overwrite existing requests and result in
            // data loss.
            if (_dataSent)
            {
                try
                {
                    // Set the _dataSent flag to false so we don't send any more
                    // data until it has been verified that this data was sent.
                    _dataSent = false;

                    synchronized(_db)
                    {
                        // Write out the data in the DataBuffer and reset the DataBuffer
                        _port.write(_db.getArray(), 0, _db.getArrayLength());
                        _db.reset();
                    }
                }
                catch (IOException ioex)
                {
                    // Reset _dataSent to true so we can attempt another data write
                    _dataSent = true;
                    FS28Demo.errorDialog("Failed to write data: " + ioex.toString());
                }
            }
            else
            {
                System.out.println("Can't send data right now, data will be sent after dataSent notify call.");
            }
        }
    }
    
    /**
     * @see net.rim.device.api.ui.Screen#close()
     */
    public void close()
    {
        if (_port != null)
        {
            _port.close();
        }        
        super.close();
    }
    
    ////////////////////////////////////////////////////////////
    //                  Menu Items                            //
    ////////////////////////////////////////////////////////////
        
    /**
     * Closes the screen
     */        
    private MenuItem _closeSP = new MenuItem("Close connection", 20, 20)
    {
        /**
         * @see java.lang.Runnable#run()
         */
        public void run() 
        {
            close();
        }
    };       
    /**
     * Save image / sample data
     */        
    private MenuItem _saveFile = new MenuItem("Save file", 20, 20)
    {
        /**
         * @see java.lang.Runnable#run()
         */
        public void run() 
        {
        	if( !mDataReceived )
        		return;
            saveFile();
        }
    };       
    //////////////////////////////////////////////////////////////
    //////////////////////////////////////////////////////////////		
    public boolean validateCommandData(byte[] Command, int[] Size)
    {   
    	if( Command.length != 13 || Command[0] != 0x40 || Command[12] != 0x0D )
    	{
    		mErrorMessage = "Invalid command data!";
    		return false;
    	}
    	int checksum = 0;
    	int i;
    	for( i=0; i<11; i++ )
    		checksum += Command[i];
    	if(Command[11] != (byte)checksum )
    	{
    		mErrorMessage = "Checksum error!";
    		return false;
    	}
    	checksum = 0;
    	short unsignedByte;
    	for( i=9; i>5; i--)
    	{
    		if( Command[i] < 0 )	// Java does not have unsigned type. 
    			unsignedByte = (short) (256 + Command[i]);
    		else
    			unsignedByte = Command[i];
    		checksum |= unsignedByte;
    		if( i > 6 )
    			checksum = checksum * 0x100;    		
    	}
    	Size[0] = checksum + 2;
        return true;
    }
    
    public void ConvertToHostSample(byte[] FamSample)        
    {
    	mHostSample[0] = (byte)0x9d;
    	mHostSample[1] = 0x02;
    	mHostSample[2] = 0x02;
    	mHostSample[3] = 0x02;
    	mHostSample[4] = 0x00;        
    	System.arraycopy(FamSample, 0, mHostSample, 5, 664);
    }

    private void startTimer(int timeout)
    {    	
        mTimerTask = new TimerTask() 
        {
          public void run() {
	          	  responseToFS28((byte)0x00, (byte)0x41);	//reply
	        	  enterCommandMode();	//return to Command mode         	 
	        	  FS28Demo.errorDialog("Timeout to receive data!\nPlease close the connection and connect again!");
            }
        };
    	mTimer.schedule(mTimerTask, timeout);    	
    }
    
    private void stopTimer()
    {
    	mTimerTask.cancel();    	
    }
    
    private void responseToFS28(byte command, byte flag){        	
        byte[] Response = new byte[]{0x40, 0x00, 0x00, 0x00, 0x00, 0x00,  0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x0D};
        byte checksum = 0;       
        Response[1] = command;
		Response[10] = flag;
		for(int i=0;i<11;i++)
		{
			checksum += Response[i];
		}
		Response[11] = checksum;
    	writeData(Response, 0, 13); 
    }
    
    private void enterCommandMode()
    {
        //reset variable value
    	mCommandLength = 0;
    	mbytesTotalRead = mbytesRead = 0;
        mDataMode = false;
        mbytesTotal = 153602;
    }
    
    private void showBitmap()
    {
    	if( mDataType != DATA_TYPE_RAWIMAGE)
    	{
    		_bmpFP.setBitmap(null);
    		return;
    	}
        mFileBmp = new MyBitmapFile(320, 480, mImageBuffer);
    	Bitmap bmp = Bitmap.createBitmapFromBytes(mFileBmp.toBytes(), 0, -1, 1);
    	_bmpFP.setBitmap(bmp);
    }
    
    private void saveFile()
    {
    	String strExtension = null;     		
    	if( mDataType == DATA_TYPE_RAWIMAGE )
    		strExtension = "bmp";
    	else if( mDataType == DATA_TYPE_WSQIMAGE )
    		strExtension = "wsq";
    	else if( mDataType == DATA_TYPE_SAMPLE )
    		strExtension = "bin";
    		
        FileSelectorPopupScreen fps = new FileSelectorPopupScreen(null, strExtension);
        fps.pickFile();
        String theFile = fps.getFile();
        if (theFile != null)
        {
        	try
        	{   	
        		FileConnection fc = (FileConnection)Connector.open( theFile, Connector.READ_WRITE);
        		if (!fc.exists())
                    fc.create();
        		OutputStream outStream = fc.openOutputStream();
        		
            	if( mDataType == DATA_TYPE_RAWIMAGE )
            	{
            		if( mFileBmp != null )
            			outStream.write(mFileBmp.toBytes());
            	}            	
            	else if( mDataType == DATA_TYPE_WSQIMAGE )
            	{
            		if(mWSQDataSize > 0)
            			outStream.write(mImageBuffer, 0, mWSQDataSize);
            	}
            	else if( mDataType == DATA_TYPE_SAMPLE )
            	{
            		ConvertToHostSample(mImageBuffer);
            		outStream.write(mHostSample);
            	}
                outStream.close();
                fc.close();
                _lblMessage.setText("File is saved to " + theFile);
        	}
        	catch(Exception ex)
        	{
        		Dialog.alert("Unable to save file. " + ex.toString());
        	}
        }
    }

}
